import io
import os
import base64
import argparse
import pathlib
import tempfile
from groq import Groq
from typing import List
from wand.image import Image

def render_page(
    src_pdf: pathlib.Path,
    fmt: str = "jpeg",
    dpi: int = 200,
    quality: int = 95
) -> List[str]:
    with Image(filename=str(src_pdf), resolution=dpi, depth=8, colorspace="gray") as images:
        base64_images = []
        for image in images.sequence:
            image = Image(image=image)
            image.sequence
            image.format = fmt.upper()
            image.type = "grayscale"
            if fmt == "jpeg":
                image.compression_quality = quality
            buf = io.BytesIO()
            image.save(file=buf)
            base64_images.append(base64.b64encode(buf.getvalue()).decode())
        return base64_images

def OCR_image (base64_image: str) -> str:
    client = Groq(api_key=os.environ.get("GROQ_API_KEY"))

    chat_completion = client.chat.completions.create(
        messages=[
            {
                "role": "user",
                "content": [
                    {
                        "type": "text",
                        "text": "Return the plain text content of the JPEG image verbatim."
                    },
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": f"data:image/jpeg;base64,{base64_image}",
                        },
                    },
                ],
            }
        ],
        model="meta-llama/llama-4-maverick-17b-128e-instruct",
    )

    completion = chat_completion.choices[0].message.content
    return completion if completion else ""

def main():
    ap = argparse.ArgumentParser(description="Split PDF and render pages to OCR-friendly images")
    ap.add_argument("pdf", help="input PDF file")
    ap.add_argument("--out-dir", "-o", type=pathlib.Path, help="output folder (default: ./pdfname_imgs)")
    ap.add_argument("--fmt", choices=["png", "jpeg"], default="png", help="image format (default: png)")
    ap.add_argument("--dpi", type=int, default=200, help="resolution (default: 200)")
    ap.add_argument("--quality", type=int, default=95, help="JPEG quality 1-100 (default: 95)")
    args = ap.parse_args()

    src_pdf = pathlib.Path(args.pdf).expanduser().resolve()
    out_dir = args.out_dir
    out_dir.mkdir(exist_ok=True)

    with tempfile.TemporaryDirectory() as temp:
        temp = pathlib.Path(out_dir)

        base64_images = render_page(src_pdf, args.fmt, args.dpi, args.quality)
        with (out_dir / src_pdf.with_suffix("").with_suffix(".txt").name).open("a") as file:
            for base64_image in base64_images:
                file.write(OCR_image(base64_image))
    
if __name__ == "__main__":
    main()
